#!/usr/bin/env python3
# Python script for generating a demo geospatial web app
#
# AUTHOR:       Huidae Cho
# COPYRIGHT:    (C) 2021 by Huidae Cho
# LICENSE:      GNU Affero General Public License v3

from bottle import route, default_app, template, request
import geopandas as gpd
from shapely.geometry import Point
import matplotlib.pyplot as plt
import io
import re
from os import path

ga_fn = 'Counties_Georgia.shp'
pt_fn = 'Points.shp'
ga_data = gpd.read_file(ga_fn)

def find_county_data(county):
    # case-insensitive search
    # https://stackoverflow.com/a/51026904
    county_data = ga_data[ga_data['NAME10'].str.contains(county, case=False)]
    return None if len(county_data) == 0 else county_data

def add_coords(lat, lon, note):
    coords_data = gpd.GeoDataFrame(
            {'note': [note]},
            geometry=gpd.GeoSeries(Point(lon, lat), crs=ga_data.crs))
    if path.exists(pt_fn):
        coords_data.to_file(pt_fn, mode='a')
    else:
        coords_data.to_file(pt_fn)

    # join with the new point only, not with the entire points shapefile
    county_row = gpd.sjoin(coords_data, ga_data)
    if county_row.empty:
        return None

    return re.sub('^[0-9 ]*', '', county_row.NAME10.to_string())

def calc_pop_dens_people_per_sqmi(county_data):
    pop = calc_pop(county_data)
    area_sqmi = calc_area_sqmi(county_data)

    return pop[0]/area_sqmi[0], f'{pop[1]}/{area_sqmi[1]}', 'population density'


def calc_pop(county_data):
    return int(county_data.totpop10), 'people', 'population'


def calc_area_sqmi(county_data):
    return float(county_data.Sq_Miles), 'sqmi', 'area'


def welcome(map_info=''):
    return template('index',
                    map_name='Georgia',
                    map_info=map_info)


def show_map(county_data, do_func):
    # https://stackoverflow.com/questions/33957720/how-to-convert-column-with-dtype-as-object-to-string-in-pandas-dataframe
    # a workaround for pandas's stringifying the entire string series object
    county = re.sub('^[0-9 ]*', '', county_data.NAME10.to_string())

    map_info = ''
    # avoid recursive calls to show_map
    if do_func != show_map:
        ret = do_func(county_data)
        map_info = f'The {ret[2]} of {county} county is {int(ret[0]):,} {ret[1]}.'

    return template('index',
                    map_name=county,
                    map_info=map_info)


@route('/')
def index():
    if 'command' not in request.query:
        return welcome()

    # strip out excessive whitespaces from the beginning and end of the
    # command, and compress whitespaces in the middle
    # https://stackoverflow.com/a/6496906
    cmd = ' '.join(request.query['command'].strip().split())

    if re.match('^add ', cmd, re.I):
        m = re.match('^[^ ]* ([+-]?[0-9]+(?:\.[0-9]*)?) ?, ?([+-]?[0-9]+(?:\.[0-9]*)?) ?(.*)$', cmd)
        if m:
            lat = float(m[1])
            lon = float(m[2])
            note = m[3]
        else:
            return welcome(f'{cmd}: Please type "add latitude, longitude note"')

        county = add_coords(lat, lon, note)
        do = 'show_map'
    else:
        # case insensitive
        cmd = cmd.lower()
        # delete non-alphanumeric characters and 'county*'
        cmd = re.sub('\'s|[^ a-z0-9]|(?:(?<![a-z])(?:county|map|is)[^ ]*)', '', cmd)
        # normalize command
        cmd = re.sub('^(.*) (population(?: density)?|area) *$',
                r'\2 \1', cmd)
        if cmd == '':
            return welcome()
        elif 'population density' in cmd:
            do = 'calc_pop_dens_people_per_sqmi'
        elif 'population' in cmd:
            do = 'calc_pop'
        elif 'area' in cmd:
            do = 'calc_area_sqmi'
        else:
            do = 'show_map'

        county = cmd.split()[-1]
        if county == 'state' or county == 'georgia':
            return welcome()

    county_data = find_county_data(county)
    if county_data is None:
        return welcome(f'{county}: No such county found')

    do_func = globals()[do]
    return show_map(county_data, do_func)


@route('/<map_name>.png')
def generate_map(map_name):
    fig, ax = plt.subplots()
    if map_name == 'Georgia':
        polygon_data = ga_data
    else:
        polygon_data = find_county_data(map_name)
    polygon_data.plot(ax=ax)
    if path.exists(pt_fn):
        pt_data = gpd.read_file(pt_fn)
        # https://geopandas.org/reference/geopandas.sjoin.html
        sel_pt_data = gpd.sjoin(polygon_data, pt_data, how='right')
        # https://stackoverflow.com/a/22553757
        sel_pt_data = sel_pt_data[sel_pt_data.index_left.notnull()]
        sel_pt_data.plot(ax=ax, color='red')

        for _,row in sel_pt_data.iterrows():
            x, y = row.geometry.coords.xy
            x = x[0]
            y = y[0]
            ax.annotate(row.note, xy=(x, y), xytext=(x, y))
#    plt.axis('off')

    data = io.BytesIO()
    plt.savefig(data, bbox_inches='tight')

    # move the file pointer to the beginnig of data
    data.seek(0)

    return data


application = default_app()
